function hess = LWM_Hessian(w_vec,resid,Q_grad,delta,vega,Gamma,eta,xi,weights)
%==========================================================================================
% This function computes the Hessian matrix of the minimization quadratic problem
% Q(theta)=(O(theta)-O_hat)'*W*(O(theta)-O_hat)/2 w.r.t. theta, where O(theta) is the mixture
% of LWM implied option prices, O_hat are the observed option prices, and 
% W = diag(weights) is the option weighting matrix
% 
%
%
% INPUT:
%        w_vec        : M-by-1 vector of weights of density components
%        resid        : N-by-1 vector of option residuals
%        Q_grad       : N-by-3M Jacobian matrix: dO(theta)/dtheta
%        delta        : N-by-M matrix of component-wise option delta
%        vega         : N-by-M matrix of component-wise option vega
%        gamma        : N-by-M matrix of component-wise option gamma
%        eta          : N-by-M matrix of component-wise option eta, where eta is defined as d^2O(theta)/dsigma^2
%        xi           : N-by-M matrix of component-wise option xi, where
%                       xi is defined as d^2O(theta)/dsigma/dF
%        weights      : N-by-1 vector of option weights. Default: equal weights
%
%
% OUTPUT:
%       hess           : 3M-by-3M Hessian matrix defined as
%                         d^2Q(theta)/dtheta/dtheta'.
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin < 9 || isempty(weights)
    weights = ones(size(reside));
end

M=length(w_vec);%number of mixtures
W = diag(weights);

%Calculate the "outerproduct of gradient part" of Hessian
hess=Q_grad'*W*Q_grad;
%Determine the range of the parameters in the parameter vector
w_range=1:M;
f_range=M+1:2*M;
sig_range=2*M+1:3*M;

%Initializing the "observation-wise Hessian" part of Hessian
hmat=zeros(3*M,3*M);
%Calculate the second derivative w.r.t. sigma_i^2
wresid = W*resid; %the weighted residual
hmat(sig_range,sig_range)=diag((w_vec'.*xi)'*wresid);
hmat(w_range,f_range)=diag(delta'*wresid);
hmat(f_range,f_range)=diag((w_vec'.*Gamma)'*wresid);
hmat(f_range,sig_range)=diag((w_vec'.*eta)'*wresid);
hmat(w_range,sig_range)=diag(vega'*wresid);

%Replicate the lower triangular half to form a symmetric matrix
hmat=ivech(vech(hmat'));
%Combine the two parts of Hessian
hess=-hmat+hess;
end


%========================================================================
%Utility functions from MFE toolbox of Kevin Sheppard
%========================================================================

function stackedData = vech(matrixData)
[k,~] = size(matrixData);

sel = tril(true(k));
stackedData = matrixData(sel);
end
function matrixData=ivech(stackedData)

if size(stackedData,2)>size(stackedData,1)
    stackedData=stackedData';
end

if size(stackedData,2)~=1
    error('STACKED_DATA must be a column vector.')
end

K2=size(stackedData,1);
K=(-1+sqrt(1+8*K2))/2;
if floor(K)~=K
    error(['The number of elemeents in STACKED_DATA must be conformable to' ...
        'the inverse vech operation.'])
end

matrixData=zeros(K);
pl=tril(true(K));
matrixData(pl)=stackedData;
diag_matrixData=diag(diag(matrixData));
matrixData=matrixData+matrixData'-diag_matrixData;
end
